import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";
import toast from "react-hot-toast";
import { UserType } from "@/types/UserTypes";

interface UserTypes {
	loading: "idle" | "pending" | "succeeded" | "failed";
	isFetching: "idle" | "pending" | "succeeded" | "failed";

	users: UserType[];
	user: UserType | null;
	error: string | null;
}
const initialState: UserTypes = {
	loading: "idle",
	isFetching: "idle",
	users: [],
	user: null,
	error: null,
};

export const createUser = createAsyncThunk(
	"users/create",
	async (data: any, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/users", {
				method: "POST",
				headers: {
					"Content-Type": "application/json",
				},
				body: JSON.stringify(data),
			});
			const responseData = await response.json();
			console.log("responseData:", responseData);
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			console.log("error:", error);
			return rejectWithValue(error);
		}
	}
);

export const fetchUsers = createAsyncThunk(
	"users/fetch",
	async (data, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/users", {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error);
		}
	}
);

export const fetchUserById = createAsyncThunk(
	"users/fetchById",
	async (id: string, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/users/${id}`, {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const updateUser = createAsyncThunk(
	"users/update",
	async ({ id, data }: { id: string; data: any }, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/users/${id}`, {
				method: "PUT",
				headers: {
					"Content-Type": "application/json",
				},
				body: JSON.stringify(data),
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const deleteUser = createAsyncThunk(
	"users/delete",
	async ({ id }: { id: string }, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/users/${id}`, {
				method: "DELETE",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const usersSlice = createSlice({
	name: "users",
	initialState,
	reducers: {},
	extraReducers: (builder) => {
		builder.addCase(createUser.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(createUser.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.users = action.payload;
			toast.success("User created successfully");
		});
		builder.addCase(createUser.rejected, (state, action) => {
			state.loading = "failed";
			console.log("action.payload:", action.payload);
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchUsers.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(fetchUsers.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.users = action.payload;
		});
		builder.addCase(fetchUsers.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchUserById.pending, (state) => {
			state.isFetching = "pending";
		});
		builder.addCase(fetchUserById.fulfilled, (state, action) => {
			state.isFetching = "succeeded";
			state.user = action.payload;
		});
		builder.addCase(fetchUserById.rejected, (state, action) => {
			state.isFetching = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(updateUser.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(updateUser.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.user = action.payload;
			toast.success("User updated successfully");
		});
		builder.addCase(updateUser.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(deleteUser.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(deleteUser.fulfilled, (state, action) => {
			state.loading = "succeeded";
			toast.success("User dleted successfully");
		});
		builder.addCase(deleteUser.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});
	},
});
