import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";
import toast from "react-hot-toast";

interface CategoryType {
	loading: "idle" | "pending" | "succeeded" | "failed";
	isFetching: "idle" | "pending" | "succeeded" | "failed";
	categories: any[];
	category: any | null;
	error: string | null;
}
const initialState: CategoryType = {
	loading: "idle",
	isFetching: "idle",
	categories: [],
	category: null,
	error: null,
};

export const createCategory = createAsyncThunk(
	"categories/create",
	async (data: any, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/categories", {
				method: "POST",
				headers: {
					"Content-Type": "application/json",
				},
				body: JSON.stringify(data),
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			console.log("error:", error);
			return rejectWithValue(error);
		}
	}
);

export const fetchCategories = createAsyncThunk(
	"categories/fetch",
	async (data, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/categories", {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error);
		}
	}
);

export const fetchCategoryById = createAsyncThunk(
	"categories/fetchById",
	async (id: string, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/categories/${id}`, {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const updateCategory = createAsyncThunk(
	"categories/update",
	async ({ id, data }: { id: string; data: any }, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/categories/${id}`, {
				method: "PUT",
				headers: {
					"Content-Type": "application/json",
				},
				body: JSON.stringify(data),
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const deleteCategory = createAsyncThunk(
	"categories/delete",
	async ({ id }: { id: string }, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/categories/${id}`, {
				method: "DELETE",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const categorySlice = createSlice({
	name: "categories",
	initialState,
	reducers: {},
	extraReducers: (builder) => {
		builder.addCase(createCategory.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(createCategory.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.categories = action.payload;
			toast.success("User created successfully");
		});
		builder.addCase(createCategory.rejected, (state, action) => {
			state.loading = "failed";
			console.log("action.payload:", action.payload);
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchCategories.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(fetchCategories.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.categories = action.payload;
		});
		builder.addCase(fetchCategories.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchCategoryById.pending, (state) => {
			state.isFetching = "pending";
		});
		builder.addCase(fetchCategoryById.fulfilled, (state, action) => {
			state.isFetching = "succeeded";
			state.category = action.payload;
		});
		builder.addCase(fetchCategoryById.rejected, (state, action) => {
			state.isFetching = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(updateCategory.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(updateCategory.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.category = action.payload;
			toast.success("Category updated successfully");
		});
		builder.addCase(updateCategory.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(deleteCategory.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(deleteCategory.fulfilled, (state, action) => {
			state.loading = "succeeded";
			toast.success("Category dleted successfully");
		});
		builder.addCase(deleteCategory.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});
	},
});
