"use client";
import React, { useEffect } from "react";
import { useForm } from "react-hook-form";
import { Input } from "@/components/ui/input";
import { useDispatch, useSelector } from "react-redux";
import { AppDispatch, RootState } from "@/lib/store/store";
import {
	createUser,
	fetchUserById,
	updateUser,
} from "@/lib/store/users/userSlice";
import { useRouter } from "next/navigation";
import { Button } from "@/components/ui/button";
import {
	Select,
	SelectContent,
	SelectItem,
	SelectTrigger,
	SelectValue,
	SelectGroup,
	SelectLabel,
} from "@/components/ui/select";
import { Components } from "@/components/Index";
import { data } from "@/utils/data";

interface UserFormProps {
	userId?: string;
}

const UserForm: React.FC<UserFormProps> = ({ userId }) => {
	const dispatch = useDispatch<AppDispatch>();
	const isLoading = useSelector((state: RootState) => state.users.loading);
	const isFetching = useSelector((state: RootState) => state.users.isFetching);
	const user = useSelector((state: RootState) => state.users.user);
	const router = useRouter();
	const {
		register,
		handleSubmit,
		formState: { errors, isValid },
		watch,
		setValue,
		reset,
	} = useForm();
	const formData = watch();

	useEffect(() => {
		if (userId) {
			const res = dispatch(fetchUserById(userId));
			reset(res);
		}
	}, []);

	const saveUser = async () => {
		const res = userId
			? await dispatch(updateUser({ id: userId, data: formData }))
			: await dispatch(createUser(formData));
		if (res.meta.requestStatus === "fulfilled") {
			router.push("/admin/users");
		}
	};
	return (
		<div className='max-w-sm ml-5'>
			{isFetching === "pending" ? (
				<Components.LoadingComponent />
			) : (
				<form onSubmit={handleSubmit(saveUser)}>
					<div>
						<label htmlFor='name'>Name</label>
						<Input
							id='name'
							type='text'
							defaultValue={user?.name}
							{...register("name", {
								required: true,
							})}
						/>
						{errors.name && (
							<p className='text-destructive'>
								{errors?.name?.message?.toString()}
							</p>
						)}
					</div>
					<div>
						<label htmlFor='phone'>Phone</label>
						<Input
							id='phone'
							type='number'
							defaultValue={user?.phone}
							{...register("phone", { required: true })}
						/>
						{errors.name && (
							<p className='text-destructive'>
								{errors?.name?.message?.toString()}
							</p>
						)}
					</div>
					<div>
						<label htmlFor='email'>Email</label>
						<Input
							id='email'
							type='email'
							defaultValue={user?.email}
							{...register("email", {
								required: "Email is required",
								pattern: {
									value: /^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,4}$/,
									message: "Invalid email address",
								},
							})}
						/>
						{errors.email && (
							<p className='text-destructive'>
								{errors.email.message?.toString()}
							</p>
						)}
					</div>
					<div className='flex gap-3 mt-4'>
						<Select
							defaultValue={user?.role}
							onValueChange={(value) => setValue("role", value)}
							{...register("role", { required: "Role is required" })}>
							<SelectTrigger className='w-[180px]'>
								<SelectValue placeholder='Select Role' />
							</SelectTrigger>
							<SelectContent>
								<SelectGroup>
									<SelectLabel>Role</SelectLabel>
									{data.roles.map((role) => (
										<SelectItem key={role.label} value={role.value}>
											{role.value}
										</SelectItem>
									))}
								</SelectGroup>
							</SelectContent>
						</Select>

						{errors.role && (
							<p className='text-destructive'>
								{errors.role.message?.toString()}
							</p>
						)}
						<Select
							defaultValue={user?.status}
							onValueChange={(value) => setValue("status", value)}
							{...register("status", { required: "status is required" })}>
							<SelectTrigger className='w-[180px]'>
								<SelectValue placeholder='Select Status' />
							</SelectTrigger>
							<SelectContent>
								<SelectGroup>
									<SelectLabel>Status</SelectLabel>
									<SelectItem value='ACTIVE'>ACTIVE</SelectItem>
									<SelectItem value='INACTIVE'>INACTIVE</SelectItem>
								</SelectGroup>
							</SelectContent>
						</Select>

						{errors.status && (
							<p className='text-destructive'>
								{errors.status.message?.toString()}
							</p>
						)}
					</div>
					{userId ? (
						""
					) : (
						<div>
							<label htmlFor='password'>Password</label>
							<Input
								id='password'
								type='password'
								{...register("password", {
									required: "Password is required",
									minLength: {
										value: 6,
										message: "Password must be at least 6 characters long",
									},
								})}
							/>
							{errors.password && (
								<p className='text-destructive'>
									{errors.password.message?.toString()}
								</p>
							)}
						</div>
					)}
					<Button type='submit' className='mt-4'>
						{isLoading === "pending" ? (
							<Components.LoadingComponent />
						) : userId ? (
							"Update User"
						) : (
							"Add User"
						)}
					</Button>
				</form>
			)}
		</div>
	);
};

export default UserForm;
