"use client";
import React, { useEffect, useRef, useState } from "react";
import Image from "next/image";

import {
	Select,
	SelectContent,
	SelectItem,
	SelectTrigger,
	SelectValue,
	SelectGroup,
	SelectLabel,
} from "@/components/ui/select";
import { Button } from "@/components/ui/button";
import Link from "next/link";
import { useDispatch, useSelector } from "react-redux";
import { AppDispatch, RootState } from "@/lib/store/store";
import { useForm } from "react-hook-form";
import { useRouter } from "next/navigation";
import { fetchCategories } from "@/lib/store/categories/categorySlice";
import { Components } from "@/components/Index";
import toast from "react-hot-toast";
import {
	createProduct,
	fetchProductById,
	updateProduct,
} from "@/lib/store/products/productSlice";

interface ProductFormProps {
	productId?: string;
}

const ProductForm: React.FC<ProductFormProps> = ({ productId }) => {
	const loading = useSelector((state: RootState) => state.products.loading);
	const product = useSelector((state: RootState) => state.products.product);

	const dispatch = useDispatch<AppDispatch>();
	const router = useRouter();
	const categories = useSelector(
		(state: RootState) => state.categories.categories
	);
	const isLoading = useSelector((state: RootState) => state.categories.loading);
	// const router = useRouter();

	const [fileName, setFileName] = useState("");
	const fileInputRef = useRef(null);
	const [fileUrl, setFileUrl] = useState("");
	const {
		handleSubmit,
		register,
		watch,
		setValue,
		reset,
		formState: { errors, isValid },
	} = useForm();
	const formData = watch();

	useEffect(() => {
		if (productId) {
			const res = dispatch(fetchProductById(productId));
			reset(res);
		}
	}, []);

	useEffect(() => {
		dispatch(fetchCategories());
	}, []);

	const handleDragOver = (e: React.DragEvent<HTMLDivElement>) => {
		e.preventDefault();
	};

	const handleDrop = (e: React.DragEvent<HTMLDivElement>) => {
		e.preventDefault();
		const files = e.dataTransfer.files;
		if (files.length) {
			setFileName(files[0].name);
			setFileUrl(URL.createObjectURL(files[0]));
			setValue("image", files[0]); // Set the file in the form data
		}
	};

	const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
		const files = e.target.files;
		if (files && files.length) {
			setFileName(files[0].name);
			setFileUrl(URL.createObjectURL(files[0]));
			setValue("image", files[0]); // Set the file in the form data
		}
	};

	const cancel = () => {
		//clear the formData
		setFileName("");
		setFileUrl("");
	};

	const handleBrowseClick = () => {
		(fileInputRef.current as HTMLInputElement | null)?.click();
	};

	const handleAddProduct = async (data: any) => {
		const formDataToSend = new FormData();
		formDataToSend.append("name", data.name);
		formDataToSend.append("priceInCents", data.priceInCents.toString());
		formDataToSend.append("quantity", data.quantity.toString());
		formDataToSend.append("description", data.description);
		formDataToSend.append(
			"isAvailableForPurchase",
			data.isAvailableForPurchase.toString()
		);
		formDataToSend.append("isFeature", data.isFeature.toString());
		formDataToSend.append("categoryId", data.categoryId);
		if (formData.image) {
			formDataToSend.append("image", formData.image);
		}

		const response = await fetch("/api/products", {
			method: productId ? "PUT" : "POST",
			body: formDataToSend,
		});
		if (response.ok) {
			router.push("/admin/products");
			window.location.reload();
		} else {
			console.log("error:", response);
			toast.error(response.statusText);
		}
	};

	return (
		<>
			<form
				className='w-full max-w-lg'
				onSubmit={handleSubmit(handleAddProduct)}
				encType='multipart/form-data' // Add this attribute
			>
				<div className='mb-4'>
					<label
						className='block text-gray-700 text-sm font-bold mb-2'
						htmlFor='name'>
						Name
					</label>
					<input
						className='shadow appearance-none border rounded w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:shadow-outline'
						id='name'
						type='text'
						defaultValue={product?.name}
						{...register("name", { required: true })}
					/>
					{errors.name && (
						<p className='text-red-500 text-xs italic'>Please enter a name.</p>
					)}
				</div>
				<div className='mb-4'>
					<label
						className='block text-gray-700 text-sm font-bold mb-2'
						htmlFor='priceInCents'>
						Price (in cents)
					</label>
					<input
						className='shadow appearance-none border rounded w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:shadow-outline'
						id='priceInCents'
						type='number'
						defaultValue={product?.priceInCents}
						{...register("priceInCents", { required: true })}
					/>
					{errors.priceInCents && (
						<p className='text-red-500 text-xs italic'>Please enter a price.</p>
					)}
				</div>
				<div className='mb-4'>
					<label
						className='block text-gray-700 text-sm font-bold mb-2'
						htmlFor='quantity'>
						Quantity
					</label>
					<input
						className='shadow appearance-none border rounded w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:shadow-outline'
						id='quantity'
						type='number'
						defaultValue={product?.quantity}
						{...register("quantity", { required: true })}
					/>
					{errors.quantity && (
						<p className='text-red-500 text-xs italic'>
							Please enter a quantity.
						</p>
					)}
				</div>
				<div className='mb-4'>
					<label
						className='block text-gray-700 text-sm font-bold mb-2'
						htmlFor='description'>
						Description
					</label>
					<textarea
						className='shadow appearance-none border rounded w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:shadow-outline'
						id='description'
						value={product?.description}
						defaultValue={product?.description}
						{...register("description", { required: true })}
					/>
					{errors.description && (
						<p className='text-red-500 text-xs italic'>
							Please enter a description.
						</p>
					)}
				</div>
				<div className='mb-4'>
					<label
						className='block text-gray-700 text-sm font-bold mb-2'
						htmlFor='isAvailableForPurchase'>
						Available for Purchase
					</label>
					<input
						className='mr-2 leading-tight'
						id='isAvailableForPurchase'
						type='checkbox'
						checked={product?.isAvailableForPurchase ? true : false}
						{...register("isAvailableForPurchase")}
					/>
				</div>
				<div className='mb-4'>
					<label
						className='block text-gray-700 text-sm font-bold mb-2'
						htmlFor='isFeature'>
						Feature Product
					</label>
					<input
						className='mr-2 leading-tight'
						id='isFeature'
						type='checkbox'
						checked={product?.isFeature ? true : false}
						{...register("isFeature")}
					/>
				</div>
				<div className='mb-4'>
					<Select
						onValueChange={(value) => setValue("categoryId", value)}
						{...register("categoryId", { required: "Category is required" })}>
						<SelectTrigger className='w-[180px]'>
							<SelectValue placeholder='Select Category' />
						</SelectTrigger>
						<SelectContent>
							<SelectGroup defaultValue={product?.categoryId}>
								<SelectLabel>
									{isLoading === "pending" ? (
										<Components.LoadingComponent />
									) : (
										"Category"
									)}
								</SelectLabel>
								{categories.map((category) => (
									<SelectItem key={category.name} value={category.id}>
										{category.name}
									</SelectItem>
								))}
							</SelectGroup>
						</SelectContent>
					</Select>

					{errors.categoryId && (
						<p className='text-red-500 text-xs italic'>
							Please select a category ID.
						</p>
					)}
				</div>
				<div
					className={`border-2 border-dashed border-tertiary/25 rounded-lg p-6 text-center`}
					onDragOver={handleDragOver}
					onDrop={handleDrop}>
					<p className='text-gray-500 mb-4'>
						{fileName || "Drag and drop your file here"}
					</p>
					<p className='text-gray-500 mb-4'>or</p>
					<button
						onClick={handleBrowseClick}
						className='bg-black text-white px-4 py-2 rounded-md hover:bg-gray-800'>
						Browse Files
					</button>
					<input
						type='file'
						className={`hidden`}
						{...register("image", { required: true })}
						onChange={(e) => {
							handleFileChange(e);
						}}
						ref={fileInputRef}
					/>
					{errors.image && (
						<p className='mt-2 mb-2 text-xs text-red-600 font-bold'>
							Image is required.
						</p>
					)}
					{fileUrl ||
						(product?.image && (
							<Image
								src={fileUrl}
								width={0}
								height={200}
								alt={fileName}
								style={{ width: "100%" }}
								className='mt-5 border border-tertiary/25 rounded-lg'
							/>
						))}
				</div>
				<div className='flex gap-4 mt-4'>
					<Button variant={"outline"} onClick={() => cancel()}>
						Cancel
					</Button>
					<Button type='submit' disabled={loading === "pending" ? true : false}>
						{productId ? "Update Product" : "Add Product"}
					</Button>
				</div>
			</form>
		</>
	);
};
export default ProductForm;
