import db from "@/utils/db";
import { NextRequest, NextResponse } from "next/server";
import { promises as fs } from "fs";
import path from "path";
import { v4 as uuidv4 } from "uuid";

export async function POST(req: NextRequest) {
	const formData = await req.formData();
	const name = formData.get("name") as string;
	const image = formData.get("image") as File;
	const priceInCents = parseInt(formData.get("priceInCents") as string, 10);
	const quantity = parseInt(formData.get("quantity") as string, 10);
	const description = formData.get("description") as string;
	const isAvailableForPurchase =
		formData.get("isAvailableForPurchase") === "true";
	const isFeature = formData.get("isFeature") === "true";
	const categoryId = formData.get("categoryId") as string;

	const user = await db.product.findFirst({
		where: { name: name },
	});
	if (user) {
		return NextResponse.json(
			{ error: "Product already exists" },
			{ status: 400 }
		);
	} else {
		try {
			// Save the image to the public folder
			const imageBuffer = Buffer.from(await image.arrayBuffer());
			const imageName = `${uuidv4()}-${image.name}`;
			const imagePath = path.join(
				process.cwd(),
				"public",
				"images",
				"products",
				imageName
			);
			await fs.writeFile(imagePath, imageBuffer);

			const product = await db.product.create({
				data: {
					name,
					priceInCents,
					quantity,
					description,
					isAvailableForPurchase,
					isFeature,
					categoryId,
					image: `/images/products/${imageName}`,
				},
			});

			return NextResponse.json(
				{ message: "product created successfuly", product },
				{ status: 200 }
			);
		} catch (error) {
			return NextResponse.json(
				{ error: "Failed to create product" },
				{ status: 500 }
			);
		}
	}
}
export async function GET(req: NextRequest) {
	const category = await db.product.findMany({
		select: {
			id: true,
			name: true,
			priceInCents: true,
			quantity: true,
			description: true,
			isAvailableForPurchase: true,
			image: true,
			createdAt: true,
			updatedAt: true,
			_count: { select: { orderItems: true } },
			category: { select: { name: true } },
			orderItems: { select: { quantity: true } }, // sellingPrice: true
			// _sum: {
			// 	orderItems: { sellingPrice: true },
			// },
		},
		orderBy: { createdAt: "desc" },
	});
	return NextResponse.json(category, { status: 200 });
}
