import db from "@/utils/db";
import { NextRequest, NextResponse } from "next/server";
import { v4 as uuidv4 } from "uuid";

export async function POST(req: NextRequest) {
	try {
		const data = await req.json();
		const { cart, pricePaidInCents, address, email, phone, town, name } = data;
		const status = "placed";
		// Check if customer exists, if not create a new customer
		let customer = await db.customer.findUnique({
			where: { phone },
		});

		if (!customer) {
			customer = await db.customer.create({
				data: {
					id: uuidv4(),
					name,
					email,
					phone,
					address,
					town,
				},
			});
		}
		const customerId = customer.id;
		// Create a new order
		const newOrder = await db.order.create({
			data: {
				id: uuidv4(),
				customerId,
				pricePaidInCents: pricePaidInCents,
				status: status,
				orderItems: {
					create: cart.map((item: any) => ({
						productId: item.id,
						quantity: item.quantity,
						sellingPrice: item.priceInCents,
					})),
				},
			},
		});

		// Deduct the quantity from the product
		for (const item of cart) {
			await db.product.update({
				where: { id: item.id },
				data: {
					quantity: {
						decrement: item.quantity,
					},
				},
			});
		}

		return NextResponse.json(newOrder, { status: 201 });
	} catch (error) {
		console.error("Error creating order:", error);
		return NextResponse.json(
			{ error: "Error creating order" },
			{ status: 500 }
		);
	}
}
export async function PUT(req: NextRequest) {
	try {
		const data = await req.json();
		const { orderId, items } = data;

		// Update the order
		const updatedOrder = await db.order.update({
			where: { id: orderId },
			data: {
				updatedAt: new Date(),
				// status:
				orderItems: {
					deleteMany: {}, // Delete existing order items
					create: items.map((item: any) => ({
						productId: item.productId,
						quantity: item.quantity,
						sellingPrice: item.sellingPrice,
					})),
				},
			},
		});

		return NextResponse.json(updatedOrder, { status: 200 });
	} catch (error) {
		console.error("Error updating order:", error);
		return NextResponse.json(
			{ error: "Error updating order" },
			{ status: 500 }
		);
	}
}
export async function GET(req: NextRequest) {
	const category = await db.order.findMany({
		select: {
			id: true,
			pricePaidInCents: true,
			createdAt: true,
			updatedAt: true,
			status: true,
			_count: { select: { orderItems: true } },
			customer: {
				select: {
					id: true,
					name: true,
					email: true,
					phone: true,
					address: true,
					town: true,
				},
			},
			orderItems: {
				select: {
					quantity: true,
					sellingPrice: true,
					orderId: true,
				},
			},
		},
		orderBy: { createdAt: "desc" },
	});
	return NextResponse.json(category, { status: 200 });
}
