"use client";
import React, { useEffect } from "react";
import Image from "next/image";
import {
	Breadcrumb,
	BreadcrumbItem,
	BreadcrumbLink,
	BreadcrumbList,
	BreadcrumbPage,
	BreadcrumbSeparator,
} from "@/components/ui/breadcrumb";
import {
	Card,
	CardContent,
	CardDescription,
	CardFooter,
	CardHeader,
	CardTitle,
} from "@/components/ui/card";
import {
	Table,
	TableBody,
	TableCell,
	TableHead,
	TableHeader,
	TableRow,
} from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import {
	DropdownMenu,
	DropdownMenuCheckboxItem,
	DropdownMenuContent,
	DropdownMenuItem,
	DropdownMenuLabel,
	DropdownMenuSeparator,
	DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Button } from "@/components/ui/button";
import Link from "next/link";
import { Icons } from "@/utils/Icons";
import AdminLayout from "../layout";
import { useDispatch, useSelector } from "react-redux";
import { AppDispatch, RootState } from "@/lib/store/store";
import {
	deleteProduct,
	fetchProducts,
} from "@/lib/store/products/productSlice";
import { AdminComponents } from "@/components/admin/Index";
import { formartDate } from "@/utils/formartDate";
import { exportToExcel } from "@/utils/exportToExcel";
import { useRouter } from "next/navigation";

export default function Products() {
	const dispatch = useDispatch<AppDispatch>();
	const products = useSelector((state: RootState) => state.products.products);
	const isLoading = useSelector((state: RootState) => state.products.loading);
	const router = useRouter();
	useEffect(() => {
		dispatch(fetchProducts());
	}, []);

	const deleteProducts = async (id: string) => {
		const data = await dispatch(deleteProduct({ id }));
		if (data.meta.requestStatus === "fulfilled") {
			router.push("/admin/products");
			window.location.reload();
		}
	};

	return (
		<AdminLayout>
			<div className='flex flex-col sm:gap-4 sm:py-4 sm:pl-14'>
				<header className='sticky top-0 z-30 flex h-14 items-center gap-4 border-b bg-background px-4 sm:static sm:h-auto sm:border-0 sm:bg-transparent sm:px-6'>
					<Breadcrumb className='hidden md:flex'>
						<BreadcrumbList>
							<BreadcrumbItem>
								<BreadcrumbLink asChild>
									<Link href='/admin'>Dashboard</Link>
								</BreadcrumbLink>
							</BreadcrumbItem>
							<BreadcrumbSeparator />
							<BreadcrumbItem>
								<BreadcrumbLink asChild>
									<Link href='/admin/Products'>Products</Link>
								</BreadcrumbLink>
							</BreadcrumbItem>
							<BreadcrumbSeparator />
							<BreadcrumbItem>
								<BreadcrumbPage>All</BreadcrumbPage>
							</BreadcrumbItem>
						</BreadcrumbList>
					</Breadcrumb>
				</header>
				<div className='flex items-center'>
					<div className='ml-auto flex items-center gap-2'>
						<DropdownMenu>
							<DropdownMenuTrigger asChild>
								<Button variant='outline' size='sm' className='h-8 gap-1'>
									<Icons.ListFilter className='h-3.5 w-3.5' />
									<span className='sr-only sm:not-sr-only sm:whitespace-nowrap'>
										Filter
									</span>
								</Button>
							</DropdownMenuTrigger>
							<DropdownMenuContent align='end'>
								<DropdownMenuLabel>Filter by</DropdownMenuLabel>
								<DropdownMenuSeparator />
								<DropdownMenuCheckboxItem checked>
									Active
								</DropdownMenuCheckboxItem>
								<DropdownMenuCheckboxItem>Draft</DropdownMenuCheckboxItem>
								<DropdownMenuCheckboxItem>Archived</DropdownMenuCheckboxItem>
							</DropdownMenuContent>
						</DropdownMenu>
						<Button
							size='sm'
							variant='outline'
							className='h-8 gap-1'
							onClick={() =>
								exportToExcel(products, "products", [
									"id",
									"name",
									"image",
									"priceInCents",
									"isAvailableForPurchase",
									"category.name",
									"createdAt",
									"updatedAt",
								])
							}>
							<Icons.File className='h-3.5 w-3.5' />
							<span className='sr-only sm:not-sr-only sm:whitespace-nowrap'>
								Export
							</span>
						</Button>
						<Link href={"/admin/products/add"}>
							<Button size='sm' className='h-8 gap-1'>
								<Icons.PlusCircle className='h-3.5 w-3.5' />
								<span className='sr-only sm:not-sr-only sm:whitespace-nowrap'>
									Add Product
								</span>
							</Button>{" "}
						</Link>
					</div>
				</div>
				{isLoading === "pending" ? (
					<AdminComponents.LoginComponent />
				) : (
					<Card x-chunk='dashboard-06-chunk-0'>
						<CardHeader>
							<CardTitle>Products</CardTitle>
							<CardDescription>
								Manage your products and view their sales performance.
							</CardDescription>
						</CardHeader>
						<CardContent>
							<Table>
								<TableHeader>
									<TableRow>
										<TableHead className='hidden w-[100px] sm:table-cell'>
											Image
										</TableHead>
										<TableHead>Name</TableHead>
										<TableHead>Category</TableHead>
										<TableHead>Available</TableHead>
										<TableHead className='hidden md:table-cell'>
											Price
										</TableHead>
										<TableHead className='hidden md:table-cell'>
											Total Sales
										</TableHead>
										<TableHead className='hidden md:table-cell'>
											Created at
										</TableHead>
										<TableHead>Actions</TableHead>
									</TableRow>
								</TableHeader>
								<TableBody>
									{products.map((product) => (
										<TableRow key={product.id}>
											<TableCell className='hidden sm:table-cell'>
												<Image
													alt={product.name}
													className='aspect-square rounded-md object-cover'
													height='64'
													src={product.image}
													width='64'
												/>
											</TableCell>
											<TableCell className='font-medium'>
												{product.name}
											</TableCell>
											<TableCell>{product.category.name}</TableCell>
											<TableCell>
												{product.isAvailableForPurchase ? (
													<Badge variant='secondary'>Yes</Badge>
												) : (
													<Badge variant='outline'>No</Badge>
												)}
											</TableCell>

											<TableCell className='hidden md:table-cell'>
												KES {product.priceInCents}
											</TableCell>
											<TableCell className='hidden md:table-cell'>
												{product._count?.orderItems}
											</TableCell>
											<TableCell className='hidden md:table-cell'>
												{formartDate(product.createdAt)}
											</TableCell>
											<TableCell>
												<DropdownMenu>
													<DropdownMenuTrigger asChild>
														<Button
															aria-haspopup='true'
															size='icon'
															variant='ghost'>
															<Icons.MoreHorizontal className='h-4 w-4' />
															<span className='sr-only'>Toggle menu</span>
														</Button>
													</DropdownMenuTrigger>
													<DropdownMenuContent align='end'>
														<DropdownMenuItem>
															<Link href={`/admin/products/${product.id}`}>
																Edit
															</Link>
														</DropdownMenuItem>
														<DropdownMenuItem
															onClick={() => deleteProducts(product.id)}>
															Delete
														</DropdownMenuItem>
													</DropdownMenuContent>
												</DropdownMenu>
											</TableCell>
										</TableRow>
									))}
								</TableBody>
							</Table>
						</CardContent>
						{/* <CardFooter>
									<div className='text-xs text-muted-foreground'>
										Showing <strong>1-10</strong> of <strong>32</strong>{" "}
										products
									</div>
								</CardFooter> */}
					</Card>
				)}
			</div>
		</AdminLayout>
	);
}
