import { ProductTypes } from "@/types/ProductTypes";
import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";
import toast from "react-hot-toast";

interface ProductType {
	loading: "idle" | "pending" | "succeeded" | "failed";
	isFetching: "idle" | "pending" | "succeeded" | "failed";
	products: ProductTypes[];
	product: any | null;
	error: string | null;
}
const initialState: ProductType = {
	loading: "idle",
	isFetching: "idle",
	products: [],
	product: null,
	error: null,
};

export const createProduct = createAsyncThunk(
	"products/create",
	async (data: any, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/products", {
				method: "POST",
				headers: {
					"Content-Type": "application/json",
				},
				body: JSON.stringify(data),
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			console.log("error:", error);
			return rejectWithValue(error);
		}
	}
);

export const fetchProducts = createAsyncThunk(
	"products/fetch",
	async (data, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/products", {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error);
		}
	}
);

export const fetchProductById = createAsyncThunk(
	"products/fetchById",
	async (id: string, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/products/${id}`, {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const updateProduct = createAsyncThunk(
	"products/update",
	async ({ id, data }: { id: string; data: any }, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/products/${id}`, {
				method: "PUT",
				headers: {
					"Content-Type": "application/json",
				},
				body: JSON.stringify(data),
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const deleteProduct = createAsyncThunk(
	"products/delete",
	async ({ id }: { id: string }, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/products/${id}`, {
				method: "DELETE",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const productSlice = createSlice({
	name: "products",
	initialState,
	reducers: {},
	extraReducers: (builder) => {
		builder.addCase(createProduct.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(createProduct.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.products = action.payload;
			toast.success("Product created successfully");
		});
		builder.addCase(createProduct.rejected, (state, action) => {
			state.loading = "failed";
			console.log("action.payload:", action.payload);
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchProducts.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(fetchProducts.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.products = action.payload;
		});
		builder.addCase(fetchProducts.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchProductById.pending, (state) => {
			state.isFetching = "pending";
		});
		builder.addCase(fetchProductById.fulfilled, (state, action) => {
			state.isFetching = "succeeded";
			state.product = action.payload;
		});
		builder.addCase(fetchProductById.rejected, (state, action) => {
			state.isFetching = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(updateProduct.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(updateProduct.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.product = action.payload;
			toast.success("Product updated successfully");
		});
		builder.addCase(updateProduct.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(deleteProduct.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(deleteProduct.fulfilled, (state, action) => {
			state.loading = "succeeded";
			toast.success("Product deleted successfully");
		});
		builder.addCase(deleteProduct.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});
	},
});
