import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";
import toast from "react-hot-toast";

interface OrderType {
	loading: "idle" | "pending" | "succeeded" | "failed";
	isFetching: "idle" | "pending" | "succeeded" | "failed";
	orders: any[];
	order: any | null;
	error: string | null;
}
const initialState: OrderType = {
	loading: "idle",
	isFetching: "idle",
	orders: [],
	order: null,
	error: null,
};

export const createOrder = createAsyncThunk(
	"orders/create",
	async (data: any, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/orders", {
				method: "POST",
				headers: {
					"Content-Type": "application/json",
				},
				body: JSON.stringify(data),
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			console.log("error:", error);
			return rejectWithValue(error);
		}
	}
);

export const fetchOrders = createAsyncThunk(
	"orders/fetch",
	async (data, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/orders", {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error);
		}
	}
);

export const fetchOrderById = createAsyncThunk(
	"orders/fetchById",
	async (id: string, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/orders/${id}`, {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const orderSlice = createSlice({
	name: "orders",
	initialState,
	reducers: {},
	extraReducers: (builder) => {
		builder.addCase(createOrder.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(createOrder.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.orders = action.payload;
			toast.success("Order Placed successfully");
		});
		builder.addCase(createOrder.rejected, (state, action) => {
			state.loading = "failed";
			console.log("action.payload:", action.payload);
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchOrders.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(fetchOrders.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.orders = action.payload;
		});
		builder.addCase(fetchOrders.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchOrderById.pending, (state) => {
			state.isFetching = "pending";
		});
		builder.addCase(fetchOrderById.fulfilled, (state, action) => {
			state.isFetching = "succeeded";
			state.order = action.payload;
		});
		builder.addCase(fetchOrderById.rejected, (state, action) => {
			state.isFetching = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});
	},
});
