import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";
import toast from "react-hot-toast";

interface OrderType {
	loading: "idle" | "pending" | "succeeded" | "failed";
	isFetching: "idle" | "pending" | "succeeded" | "failed";
	customers: any[];
	customer: any | null;
	error: string | null;
}
const initialState: OrderType = {
	loading: "idle",
	isFetching: "idle",
	customers: [],
	customer: null,
	error: null,
};

export const createCustomer = createAsyncThunk(
	"customers/create",
	async (data: any, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/customers", {
				method: "POST",
				headers: {
					"Content-Type": "application/json",
				},
				body: JSON.stringify(data),
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			console.log("error:", error);
			return rejectWithValue(error);
		}
	}
);

export const fetchCustomers = createAsyncThunk(
	"customers/fetch",
	async (data, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/customers", {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error);
		}
	}
);

export const fetchCustomerById = createAsyncThunk(
	"customers/fetchById",
	async (id: string, { rejectWithValue }) => {
		try {
			const response = await fetch(`/api/customers/${id}`, {
				method: "GET",
				headers: {
					"Content-Type": "application/json",
				},
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}

			return responseData;
		} catch (error: any) {
			return rejectWithValue(error.message);
		}
	}
);

export const customerSlice = createSlice({
	name: "customers",
	initialState,
	reducers: {},
	extraReducers: (builder) => {
		builder.addCase(createCustomer.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(createCustomer.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.customers = action.payload;
			toast.success("customers created successfully");
		});
		builder.addCase(createCustomer.rejected, (state, action) => {
			state.loading = "failed";
			console.log("action.payload:", action.payload);
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchCustomers.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(fetchCustomers.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.customers = action.payload;
		});
		builder.addCase(fetchCustomers.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});

		builder.addCase(fetchCustomerById.pending, (state) => {
			state.isFetching = "pending";
		});
		builder.addCase(fetchCustomerById.fulfilled, (state, action) => {
			state.isFetching = "succeeded";
			state.customer = action.payload;
		});
		builder.addCase(fetchCustomerById.rejected, (state, action) => {
			state.isFetching = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});
	},
});
