import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";
import Cookies from "js-cookie";
import toast from "react-hot-toast";
import { jwtDecode } from "jwt-decode";

interface AuthUser {
	loading: "idle" | "pending" | "succeeded" | "failed";
	user: any | null;
	token: string | null;
	error: string | null;
}
const initialState: AuthUser = {
	loading: "idle",
	user: null,
	token: null,
	error: null,
};

export const login = createAsyncThunk(
	"auth/login",
	async (data: any, { rejectWithValue }) => {
		try {
			const response = await fetch("/api/auth/login", {
				method: "POST",
				headers: {
					"Content-Type": "application/json",
				},
				body: JSON.stringify(data),
			});
			const responseData = await response.json();
			if (!response.ok) {
				return rejectWithValue(responseData.error);
			}
			Cookies.set("token", responseData.access);
			const decodedToken = jwtDecode(responseData.access);
			Cookies.set("authUser", JSON.stringify(decodedToken));
			return responseData;
		} catch (error: any) {
			return rejectWithValue(error);
		}
	}
);
export const authSlice = createSlice({
	name: "auth",
	initialState,
	reducers: {
		checkToken: (state) => {
			const localToken = localStorage.getItem("token");
			const stateToken = state.token;
			state.token = stateToken || localToken;
		},
		autoLogout: (state) => {
			state.token = null;
			Cookies.remove("token");
			Cookies.remove("authUser");
		},
	},
	extraReducers: (builder) => {
		builder.addCase(login.pending, (state) => {
			state.loading = "pending";
		});
		builder.addCase(login.fulfilled, (state, action) => {
			state.loading = "succeeded";
			state.user = action.payload.user;
			toast.success("Login successfull");
			localStorage.setItem("token", action.payload.access);
			localStorage.setItem("authUser", JSON.stringify(action.payload.user));
			state.token = action.payload.access;
		});
		builder.addCase(login.rejected, (state, action) => {
			state.loading = "failed";
			state.error = action.payload as string;
			toast.error(action.payload as string);
		});
	},
});

export const { checkToken, autoLogout } = authSlice.actions;
