import Orders from "@/app/admin/orders/page";
import db from "@/utils/db";
import { NextRequest, NextResponse } from "next/server";
import { v4 as uuidv4 } from "uuid";

export async function POST(req: NextRequest) {
	try {
		const data = await req.json();
		const { address, email, phone, town, name } = data;
		let customer = await db.customer.findUnique({
			where: { phone },
		});

		if (!customer) {
			customer = await db.customer.create({
				data: {
					id: uuidv4(),
					name,
					email,
					phone,
					address,
					town,
				},
			});
		}

		return NextResponse.json(customer, { status: 201 });
	} catch (error) {
		console.error("Error creating customer:", error);
		return NextResponse.json(
			{ error: "Error creating customer" },
			{ status: 500 }
		);
	}
}
export async function PUT(req: NextRequest) {
	try {
		const data = await req.json();
		const { customerId, address, email, phone, town, name } = data;

		// Update the customer
		const updated = await db.customer.update({
			where: { id: customerId },
			data: {
				updatedAt: new Date(),
				name,
				email,
				phone,
				address,
				town,
			},
		});

		return NextResponse.json(updated, { status: 200 });
	} catch (error) {
		console.error("Error updating order:", error);
		return NextResponse.json(
			{ error: "Error updating order" },
			{ status: 500 }
		);
	}
}
export async function GET(req: NextRequest) {
	const customers = await db.customer.findMany({
		select: {
			id: true,
			name: true,
			email: true,
			phone: true,
			address: true,
			town: true,
			createdAt: true,
			updatedAt: true,
			status: true,
			_count: { select: { orders: true } },
			orders: {
				select: {
					pricePaidInCents: true,
					status: true,
					createdAt: true,
				},
			},
		},
		orderBy: { createdAt: "desc" },
	});
	// Calculate the total pricePaidInCents for each customer
	const result = customers.map((customer) => {
		const sales = customer.orders.reduce(
			(sum, order) => sum + order.pricePaidInCents,
			0
		);
		return {
			...customer,
			sales,
		};
	});
	return NextResponse.json(result, { status: 200 });
}
